/**
* \file: errmemd_backend_buffered.c
*
* Definition of a backend which represents a simple buffer.
* 
* Definition of the functions provided by the buffer backend.
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "errmem_backend.h"

static inline const char * kind(int type)
{
	switch(type) {
		case ERRMEM_TYPE_TRACE:
			return "TRACE";
		case ERRMEM_TYPE_ASCII:
			return "ASCII";
	}

	return "unknown";
}
// store - log an error memory message
static unsigned store(struct ErrmemBackend* backend, int len,
		  struct errmem_message* msg)
{
	ErrmemBackendBuffered_t* l = (ErrmemBackendBuffered_t*) backend;
	unsigned long long t;
	unsigned mask = ~0;

	// helper macros to print message flags and type
#define flag(s) (msg->internal.flags & mask & ERRMEM_FLAG_##s ? " " #s : "")

#define ERRMEM_FLAG_REBOOT     ERRMEM_FLAG_RESTART
#define ERRMEM_FLAG_POWER      ERRMEM_FLAG_INITIALIZED

	memcpy(&t, &msg->internal.local_clock, sizeof (t));

	// mark power-up and reboot
	*l->buffer = '\0';
	if (msg->internal.flags & ERRMEM_FLAG_POWER)
		sprintf(l->buffer, "= POWER UP =====================================\n");
	else if (msg->internal.flags & ERRMEM_FLAG_REBOOT)
		sprintf(l->buffer, "- REBOOT ---------------------------------------\n");

	if (msg->type!=ERRMEM_TYPE_TRACE &&
	    !msg->internal.offset) {
		sprintf(l->buffer+strlen(l->buffer), "{%05u}",
			(unsigned) msg->internal.seqnum);

		if (msg->message[0]!='[')
			// sequence number and timestamp
			sprintf(l->buffer+strlen(l->buffer), "[%5lu.%06lu] ",
				(unsigned long) (t/1000000000LL),
				(unsigned long) (t%1000000000LL)/1000);
	}

	if ((msg->type!=ERRMEM_TYPE_TRACE &&
	     msg->type!=ERRMEM_TYPE_ASCII) ||
		/* PRQA: Lint Message 648: We explicitly want to create a mask*/
		/*lint -save -e648 */
	    (msg->internal.flags & ERRMEM_FLAG_ANY_ERROR))
		/*lint -restore */
		// show errors
		sprintf(l->buffer+strlen(l->buffer),
			"%s%s%s%s%s%s "
			"length=%d offset=%u\n",
			kind(msg->type),
			flag(DROPPED),
			flag(CRC_ERROR),
			flag(INCOMPLETE),
			flag(BUSY),
			flag(OUTDATED),
			msg->length,
			msg->internal.offset);

#undef flag
#undef kind

	if (len>0 && msg->length>0) {
		if(msg->type==ERRMEM_TYPE_ASCII) {
			/* PRQA: Lint Message 648: We explicitly want to create a mask*/
			/*lint -save -e648 */
			unsigned owner = ERRMEM_GET_VM(msg->internal.flags);
			/*lint -restore */
			// increase the buffer size as required
			// We can allocate more than necessary
			while (l->size<strlen(l->buffer)+msg->length+2 + VM_STRING_SZ) {
				l->size *= 2;
				l->buffer = realloc(l->buffer, l->size);
			}

			// show ascii messages
			switch(owner) {
			case OWNER_NONE:
				sprintf(l->buffer+strlen(l->buffer), "%s", msg->message);
				break;
			case OWNER_VMM:
				sprintf(l->buffer + strlen(l->buffer),
						VMM_STRING"%s", msg->message);
				break;
			default:
				sprintf(l->buffer + strlen(l->buffer),
						VM_STRING"%s", (owner - 2), msg->message);
			}

			if (msg->message[msg->length-1]!='\n')
				sprintf(l->buffer+strlen(l->buffer), "\n");
		} else {
			unsigned i, p=strlen(l->buffer);

			// increase the buffer size as required
			while (l->size<p+msg->length*3+2) {
				l->size *= 2;
				l->buffer = realloc(l->buffer, l->size);
			}

			// show binary messages
			for (i=0; i<msg->length; i++, p+=3)
				sprintf(l->buffer+p, "%02x ", (unsigned)
					msg->message[i]);
			sprintf(l->buffer+p-1, "\n");
		}
	}

	sprintf(l->buffer+strlen(l->buffer), "\r");
	return msg->internal.seqnum;
}

// destroy - free resources
static void destroy(struct ErrmemBackend* backend)
{
	ErrmemBackendBuffered_t* l = (ErrmemBackendBuffered_t*) backend;

	if (l->buffer)
		free(l->buffer);

	free(l);
}

static ErrmemBackendBuffered_t* create(ErrmemBackendDev_t* bdev)
{
    ErrmemBackendBuffered_t* l = 
	(ErrmemBackendBuffered_t*)calloc(1, sizeof(ErrmemBackendBuffered_t));
    if (l) {
		l->backend.store = store;
		l->backend.destroy = destroy;
		l->size = 4096;
		l->buffer = malloc(l->size);
		l->type   = bdev->type;
    }
    return l;
}

// errmem_backend_create_buffered - create a buffered error memory backend
ErrmemBackend_t* errmem_backend_create_buffered(ErrmemBackendDev_t* bdev)
{
    ErrmemBackendBuffered_t* l = create(bdev);
    if (!l || !l->buffer)
	return NULL;
    
	l->backend.next = NULL;

    return &l->backend;
}
